----------------------------------------------------------------------------------
-- Company: www.simong.se
-- Engineer: Simon Gustafsson
-- 
-- Create Date:    16:57:37 12/12/2009 
-- Design Name: 
-- Module Name:    uart_rx - rtl 
-- Project Name: 
-- Target Devices: 
-- Tool versions: 
-- Description: 
--
--   Reception UART. No majority decission to determine if bit high/low, we only sample once...
--
-- Dependencies: 
--
-- Revision: 
-- Revision 0.01 - File Created
-- Additional Comments: 
-- Copyright (c) 2009 Simon Gustafsson (www.simong.se)
----------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

---- Uncomment the following library declaration if instantiating
---- any Xilinx primitives in this code.
--library UNISIM;
--use UNISIM.VComponents.all;

entity uart_rx is
    generic( baudrate:positive:=115200;
				 F:positive:=48000000 );
    Port ( 
		resetn : in STD_LOGIC;
		clk : in std_logic;
		rxd : in  STD_LOGIC;
        uart_rxd : out  STD_LOGIC_VECTOR (7 downto 0);
        uart_rxd_full : buffer  STD_LOGIC;
        uart_clear_rxd_full : in std_logic
        );
end uart_rx;

architecture rtl of uart_rx is

--component declarations
component delays
	generic (cycles:positive:=10);
	port ( clk : in std_logic;
			 resetn : in std_logic;
			 timeout : out std_logic);
end component;

for U1:delays use entity work.delays(rtl); -- used by UART
for U2:delays use entity work.delays(rtl); -- used by UART

signal delay_timed_out1 : std_logic;
signal delay_timed_out2 : std_logic;
signal delay_resetn : std_logic;

signal uart_rxd_temp: std_logic_vector(7 downto 0);
type state_type_rxd is (wait_for_start_bit, wait_for_d0, d0, d1, d2, d3, d4, d5, d6, d7, stop_bit);
signal state_rxd:state_type_rxd;

constant bitcycles:positive := F/baudrate;
constant bitcycles_2:positive := bitcycles/2;

begin

	U1:delays generic map(cycles=>bitcycles_2) port map(clk=>clk, resetn=>delay_resetn, timeout=>delay_timed_out1  );
	U2:delays generic map(cycles=>bitcycles) port map(clk=>clk, resetn=>delay_resetn, timeout=>delay_timed_out2  );

	-- For reception of serial commands
	-- UART part (serial to paralell)
	--
	-- Sets uart_rxd_full when a byte is available in uart_rxd
	process(clk, resetn)
	begin
		if resetn='0' then
			uart_rxd_full <= '0';
			uart_rxd<=(others=>'0');
			uart_rxd_temp<=(others=>'0');
			delay_resetn<='0';
		elsif clk'event and clk='1' then
		
			if uart_clear_rxd_full='1' then
				uart_rxd_full<='0';
			end if;
			
			case state_rxd is
								
				when wait_for_start_bit => if rxd='0' then
														-- We got the start bit, 
														-- wait for 1.5 bit lengths (to center over d0)
														delay_resetn<='0';
														state_rxd <= wait_for_d0;
														--wait until clk='1';
													else
														state_rxd <= wait_for_start_bit;
													end if;
													
				when wait_for_d0 =>	
										delay_resetn<='1';
										if delay_timed_out2='1' then --one bit time
											state_rxd <= d0;
											delay_resetn<='0';
										end if;
										
													
				when stop_bit => --we are now centered over the stop bit,
									  --we could
									  -- 1) Ensure that bit is high
									  -- 2) Ensure that uart buffer not full
									  -- 3) Flag that uart has byte
									  -- 4) Change state to wait_for_start_bit
									  if delay_timed_out2='1' then
											delay_resetn <= '0';
											
											assert rxd='1'
											report "No stop bit when expected"
											severity warning;

											assert uart_rxd_full='0'
											report "UART buffer full - trowing data away"
											severity warning;
											
											uart_rxd <= uart_rxd_temp;
											uart_rxd_full <= '1';
											state_rxd <= wait_for_start_bit;
										else
											delay_resetn <= '1';
										end if;
									  
				when d0		=>   --delay more
									  if delay_timed_out1='1' then -- half bit time
											delay_resetn <= '0';
										
											uart_rxd_temp <= (rxd & "0000000");
											state_rxd<=d1;
										else
											delay_resetn <= '1';
										end if;
									  
				when others =>		--Shift data into register
										if delay_timed_out2='1' then
											delay_resetn <= '0';
										
											uart_rxd_temp <= (rxd & uart_rxd_temp(7 downto 1));
										
			
											case state_rxd is
												when d1 => state_rxd<=d2;
												when d2 => state_rxd<=d3;
												when d3 => state_rxd<=d4;
												when d4 => state_rxd<=d5;
												when d5 => state_rxd<=d6;
												when d6 => state_rxd<=d7;
												when d7 => state_rxd<=stop_bit;
												when others => state_rxd<=wait_for_start_bit;
											end case;
										else
											delay_resetn <= '1';
										end if;
										
			end case;
		end if;
	end process;



end rtl;

