----------------------------------------------------------------------------------
-- Company: www.simong.se
-- Engineer: Simon Gustafsson
-- 
-- Create Date:    19:59:23 08/25/2008 
-- Design Name: 
-- Module Name:    freq_generator - rtl 
-- Project Name: 
-- Target Devices: 
-- Tool versions: 
-- Description: 
-- 
--   A simple DDS generator - receives commands over a serial interface, and outputs 
--   waveform as 8 parallel data bits, and additionally both by PWM:ing an output,
--   as well as by delta-sigma-modulating a different output pin.
--
-- Dependencies: 
--
-- Revision: 
-- Revision 0.01 - File Created
-- Additional Comments: 
-- Copyright (c) 2009 Simon Gustafsson (www.simong.se)
----------------------------------------------------------------------------------


library IEEE;
--Library UNISIM;              -- One of two lines needed when having a DCM
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

--use UNISIM.vcomponents.all;  -- One of two lines needed when having a DCM



entity freq_generator is
    Port ( clk_ext : in std_logic;
           txd : out  STD_LOGIC;
           rxd : in  STD_LOGIC;
           waveout : out  STD_LOGIC_VECTOR (7 downto 0);
			  pwm_out : out std_logic;
			  pwm2_out : out std_logic;
			  leds : out STD_LOGIC_VECTOR (3 downto 0);
           test : in  STD_LOGIC_VECTOR (2 downto 0);
			  reset : in std_logic;
			  RS232_Rx_debug_out : out std_logic );
end freq_generator;

architecture rtl of freq_generator is

--component declarations
COMPONENT dcm1
	PORT(
		CLKIN_IN : IN std_logic;
		RST_IN : IN std_logic;          
		CLKFX_OUT : OUT std_logic;
		CLKIN_IBUFG_OUT : OUT std_logic
		);
END COMPONENT;

component uart_rx is
    generic( baudrate:positive:=115200;
				 F:positive:=48000000 );
    Port ( 
		resetn : in STD_LOGIC;
		clk : in std_logic;
		rxd : in  STD_LOGIC;
        uart_rxd : out  STD_LOGIC_VECTOR (7 downto 0);
        uart_rxd_full : buffer  STD_LOGIC;
        uart_clear_rxd_full : in std_logic
        );
end component;

component uart_tx is
    generic( baudrate:positive:=115200;
				 F:positive:=48000000 );
    Port ( 
		resetn : in STD_LOGIC;
		clk : in std_logic;
		txd : out  STD_LOGIC;
        uart_txd : in  STD_LOGIC_VECTOR (7 downto 0);
        uart_txd_empty : out  STD_LOGIC;
        uart_txd_start : in std_logic
        );
end component;

component my_ram
	generic( addr_width:positive:=8; data_width:positive:=8 );
	port(addr: in std_logic_vector(addr_width-1 downto 0);
	clk,we_n: in std_logic;
	din:in std_logic_vector(data_width-1 downto 0);
	dout:out std_logic_vector(data_width-1 downto 0));
end component;

component dac
	port(
		resetn : in std_logic;
		clk : in std_logic;
		data : in std_logic_vector(7 downto 0);
		pwm_out : out std_logic
		);
end component;

for RAM_A:my_ram use entity work.my_ram(rtl);
for RAM_B:my_ram use entity work.my_ram(rtl);

-- PWM related
signal pwm_lim  : std_logic_vector(7 downto 0);
signal pwm_cntr : std_logic_vector(8 downto 0);


-- DDS related
constant cntr_bits:positive := 32;
constant top:std_logic_vector(21 downto 0) := CONV_STD_LOGIC_VECTOR(3518437, 22);--(2**(32+16))/80e6;
signal cntr : std_logic_vector((cntr_bits-1) downto 0);
signal inc  : std_logic_vector((cntr_bits-1) downto 0);
signal inc_new_value:std_logic_vector((cntr_bits-1) downto 0);
	
-- UART related:
constant F:integer := 80000000; -- CLK frequency
constant baudrate:positive := 115200;--115200;

for UART_T:uart_tx use entity work.uart_tx(rtl);
for UART_R:uart_rx use entity work.uart_rx(rtl);

signal uart_txd: std_logic_vector(7 downto 0);
signal uart_txd_empty: std_logic; --driven by uart
signal uart_txd_start:std_logic; -- pulsed by others than the uart;

signal uart_rxd: std_logic_vector(7 downto 0);
signal uart_rxd_full: std_logic;
signal uart_clear_rxd_full:  std_logic;


-- Parser related:
type state_type_parser is (wait_for_newline, 
wait_for_F, 
F_before_decimal, 
F_after_decimal, 
F_waiting_for_mul,
W_counting,
M_getmode,
on_newline,
print_info,
print_info_I0,
print_info_I1,
print_info_I2,
print_info_I3,
print_info_I4,
print_info_I5,

print_info_hex_word,
print_info_hex_byte,
print_info_hex_byte_0,
print_info_hex_byte_1
);

signal parser_state:state_type_parser;
signal parser_state_ret:state_type_parser; --return state (when in "subprogram")
signal parser_state_arg1:std_logic_vector(7 downto 0); -- for "subprograms"
signal parser_freq:std_logic_vector(31 downto 0);
signal parser_waveform_count:std_logic_vector(7 downto 0);


--Waveform ram related
signal should_switch_waveform_bank:std_logic; --Used internally
signal ram_select:std_logic; 	--Switches between two ram banks. The one not used by the DDS is used by the UART
										--'0' means rama is used by UART, and ramb by DDS
signal rama_we_n:std_logic;
signal ramb_we_n:std_logic;
signal rama_din:std_logic_vector(7 downto 0);
signal ramb_din:std_logic_vector(7 downto 0);
signal rama_dout:std_logic_vector(7 downto 0);
signal ramb_dout:std_logic_vector(7 downto 0);
signal rama_addr:std_logic_vector(7 downto 0);
signal ramb_addr:std_logic_vector(7 downto 0);

signal ramDDS_dout:std_logic_vector(7 downto 0);
signal ramDDS_addr:std_logic_vector(7 downto 0);

signal ramUART_we_n:std_logic;
signal ramUART_din:std_logic_vector(7 downto 0);
signal ramUART_addr:std_logic_vector(7 downto 0);


--since input buttons active high
signal resetn:std_logic;

signal test_last : std_logic_vector(2 downto 0);

signal test_from_host:std_logic_vector(2 downto 0); -- Host can signal waveform mode switch...

signal clk : std_logic; -- used by the DCM as clock output, and is input for the rest of the design.

signal should_calculate_inc:std_logic;
signal done_calculate_inc:std_logic;

signal clk_cntr:std_logic_vector(3 downto 0);
signal slow_clk:std_logic;
attribute buffer_type: string;
attribute buffer_type of slow_clk: signal is "bufg"; --or bufg?

begin
	Inst_dcm1: dcm1 PORT MAP(
		CLKIN_IN => clk_ext,
		RST_IN => reset,
		CLKFX_OUT => clk,
		CLKIN_IBUFG_OUT => open
	);
	
	RS232_Rx_debug_out <= '1';

	process(clk, reset)
	begin
		resetn <= not reset;
	end process;

	-- Updates a clock counter (which we use for clocking slower functions)
	process(clk, resetn)
	begin
		if resetn='0' then
			clk_cntr<=(others=>'0');
			slow_clk<='0';
		elsif rising_edge(clk) then
			clk_cntr <= clk_cntr+1;
			slow_clk<=clk_cntr(3);
		end if;
	end process;

	process(clk, resetn)
	begin
		if resetn='0' then
			test_last <= "001";
		elsif rising_edge(clk) then
		   if test_from_host /= "000" then
				test_last <= test_from_host;
			elsif test /= "000" then 
				test_last <= test;
			end if;
		end if;
	end process;


	-- Handles calculation of new increment value for the DDS (using a slower clock)
	process(slow_clk, resetn)
	variable inc_temp:std_logic_vector((32+22-1) downto 0);
	begin
		if resetn='0' then
			done_calculate_inc<='0';
			inc_new_value <= (10=>'1', others=>'0'); -- use 12=>'1' to make it go faster...
		elsif rising_edge(slow_clk) then
			if should_calculate_inc='1' then
				inc_temp := (parser_freq*top);
				inc_new_value <= inc_temp((32+16-1) downto 16);
				done_calculate_inc<='1';
			elsif done_calculate_inc='1' then
				done_calculate_inc<='0';
			end if;
		end if;
	end process;


	DAC1:dac port map(resetn=>resetn, clk=>clk, data=>pwm_lim, pwm_out=>pwm2_out);

	UART_R:uart_rx generic map(baudrate=>baudrate, F=>F)
				port map(resetn=>resetn, clk=>clk, rxd=>rxd, uart_rxd=>uart_rxd, uart_rxd_full=>uart_rxd_full, uart_clear_rxd_full=>uart_clear_rxd_full );
	
	UART_T:uart_tx generic map(baudrate=>baudrate, F=>F)
				port map(resetn=>resetn, clk=>clk, txd=>txd, uart_txd=>uart_txd, uart_txd_empty=>uart_txd_empty, uart_txd_start=>uart_txd_start );
	

	RAM_A:my_ram	generic map(addr_width=>8, data_width=>8) 
						port map(clk=>clk, we_n=>rama_we_n, addr=>rama_addr, din=>rama_din, dout=>rama_dout);
	RAM_B:my_ram	generic map(addr_width=>8, data_width=>8) 
						port map(clk=>clk, we_n=>ramb_we_n, addr=>ramb_addr, din=>ramb_din, dout=>ramb_dout);
	
	--Make all connections between the two ram banks and UART/DDS generator...
	rama_addr <= ramUART_addr when ram_select='0' else ramDDS_addr;
	rama_din  <= ramUART_din when ram_select='0' else (others=>'0');
	rama_we_n <= ramUART_we_n when ram_select='0' else '1';
	
	ramb_addr <= ramUART_addr when ram_select='1' else ramDDS_addr;
	ramb_din  <= ramUART_din when ram_select='1' else (others=>'0');
	ramb_we_n <= ramUART_we_n when ram_select='1' else '1';
	
	
	ramDDS_dout <= ramb_dout when ram_select='0' else rama_dout;
	
	-- Main PLL loop
	process(clk,resetn)
	begin
		if resetn='0' then
			cntr<=(others=>'0');
		elsif rising_edge(clk) then
			-- Main PLL loop
			cntr <= cntr+inc;
		end if;
	end process;

	-- Loop outputting the waveform
	process(clk,resetn)
	begin
		if resetn='0' then
			waveout <= "00000000";
			pwm_lim <= "00000000";
			leds <= "0000";		
		elsif rising_edge(clk) then
			if test_last(0)='1' then
				waveout <= cntr((cntr_bits-1) downto (cntr_bits-8)); --Highest bits of counter is output -> triangular wave
				pwm_lim <= cntr((cntr_bits-1) downto (cntr_bits-8));
				leds <= cntr((cntr_bits-1) downto (cntr_bits-4));
			elsif test_last(1)='1' then
				if cntr(cntr_bits-1)='1' then
					waveout <= (others=>'1');
					pwm_lim <= (others=>'1');
					leds <= (others=>'1');
				else
					waveout <= (others=>'0');
					pwm_lim <= (others=>'0');
					leds <= (others=>'0');
				end if;
			elsif test_last(2)='1' then
					ramDDS_addr<=cntr((cntr_bits-1) downto (cntr_bits-8));
					waveout <= ramDDS_dout; -- we are outputing correct data (with one clk delay)
					pwm_lim <= ramDDS_dout;
					leds <= ramDDS_dout(7 downto 4);
			end if;
		end if;
	end process;
	
	
	-- Loop PWM:ing the eight bit data on some other signal
	process(clk, resetn)
	begin
		if resetn='0' then
			pwm_cntr<=(others=>'0');
			pwm_out<='0';
		elsif rising_edge(clk) then
			-- Main PWM loop
			pwm_cntr <= pwm_cntr + 1;
				
			if pwm_cntr(8 downto 1) > pwm_lim then
				pwm_out <= '0';
			else
				pwm_out <= '1';
			end if;
		end if;
	end process;

   -- Checks for uart_rxd_full, and parses numbers sent to generator...
	-- uses uart_rxd and uart_rxd_full as inputs
	-- parser_state as state variable, and
	-- parser_freq to build the frequency 
	process(clk,resetn)

	variable uart_rxd_full_delay:std_logic;
	
	function toHexChar(a:in std_logic_vector(3 downto 0)) return
		std_logic_vector is
	begin
		if a >= "1010" then
			return ("0000" & a) + "00110111"; -- plus 'A' minus ten
		else
			return ("0000" & a) + "00110000"; -- plus '0'
		end if;
	end toHexChar;
	
	begin
		
		if resetn='0' then
			ram_select<='0';
			--parser_state <= print_info;
			parser_state <= wait_for_F;
			parser_freq <= (others=>'0');
			inc<=(10=>'1', others=>'0'); -- use 12=>'1' to make it go faster...
			uart_clear_rxd_full <= '0';
			uart_rxd_full_delay:='0';
			uart_txd_start <= '0';
			should_switch_waveform_bank <= '0';
		elsif rising_edge(clk) then
		
			if should_switch_waveform_bank='1' then
				ram_select <= not ram_select;
				should_switch_waveform_bank <= '0';
			end if;
			
			if uart_clear_rxd_full='1' and uart_rxd_full='0' then
				uart_clear_rxd_full<='0';
			end if;
			
			-- These states handles printing of information
			-- currently only for debugging (triggered when a '?'
			-- is received...
			case parser_state is
				when print_info =>
					if uart_txd_empty='1' and uart_txd_start='0' then
						uart_txd <= "00111111";
						uart_txd_start <= '1';
						parser_state_ret <= print_info_I0;
						parser_state <= print_info_hex_byte;
						parser_state_arg1 <= inc(31 downto 24);
					else
						uart_txd_start<='0';
					end if;

				when print_info_I0 =>
					if uart_txd_empty='1' and uart_txd_start='0' then
						parser_state_ret <= print_info_I1;
						parser_state <= print_info_hex_byte;
						parser_state_arg1 <= inc(23 downto 16);
					else
						uart_txd_start<='0';
					end if;
					
				when print_info_I1 => 
					if uart_txd_empty='1' and uart_txd_start='0' then
						parser_state_ret <= print_info_I2;
						parser_state <= print_info_hex_byte;
						parser_state_arg1 <= inc(15 downto 8);
					else
						uart_txd_start<='0';
					end if;
			
				when print_info_I2 => 
					if uart_txd_empty='1' and uart_txd_start='0' then
						parser_state_ret <= print_info_I3;
						parser_state <= print_info_hex_byte;
						parser_state_arg1 <= inc(7 downto 0);
					else
						uart_txd_start<='0';
					end if;
					
				when print_info_I3 => 
					if uart_txd_empty='1' and uart_txd_start='0' then
						uart_txd <= "00111010";
						uart_txd_start <= '1';
						parser_state_ret <= print_info_I4;
						parser_state <= print_info_hex_byte;
						parser_state_arg1 <= "0011" & "0" & test_last;
					else
						uart_txd_start<='0';
					end if;
					
				when print_info_I4 =>
					if uart_txd_empty='1' and uart_txd_start='0' then
						parser_state <= print_info_I5;
					else
						uart_txd_start<='0';
					end if;
					
				when print_info_I5 =>
					if uart_txd_empty='1' and uart_txd_start='0' then
						parser_state <= wait_for_F;
					else
						uart_txd_start<='0';
					end if;
					
				
				when print_info_hex_byte =>
					if uart_txd_empty='1' and uart_txd_start='0' then
						--uart_txd <= "00110000" + ("0000" & parser_state_arg1(7 downto 4));
						uart_txd <= toHexChar(parser_state_arg1(7 downto 4));
						uart_txd_start <= '1';
						parser_state <= print_info_hex_byte_0;
					else
						uart_txd_start<='0';
					end if;
						
				when print_info_hex_byte_0 =>
					if uart_txd_empty='1' and uart_txd_start='0' then
						--uart_txd <= "00110000" + ("0000" & parser_state_arg1(3 downto 0));
						uart_txd <= toHexChar(parser_state_arg1(3 downto 0));
						uart_txd_start <= '1';
						parser_state <= parser_state_ret;
					else
						uart_txd_start<='0';
					end if;
						
				when others => null;
			end case;
			
			
			if parser_state=F_waiting_for_mul then
				-- Give multiplication some additional clock cycles
				if done_calculate_inc='1' then
					should_calculate_inc<='0';
					inc <= inc_new_value;
					parser_state <= wait_for_F;
				else
					assert should_calculate_inc='1'
					report "Might be in an infinite loop"
					severity warning;
				end if;
			elsif uart_rxd_full_delay='0' and uart_rxd_full='1' then
				case parser_state is
				
					when wait_for_newline =>
													if uart_rxd = "00001010" or uart_rxd = "00001101" then --new line
														parser_state <= wait_for_F;
													end if;
	
	
					when wait_for_F =>				if uart_rxd = "01000110" then  --was 'F'
														parser_state <= F_before_decimal;
														parser_freq <= (others =>'0');
													elsif uart_rxd = "01010111" then --was 'W'
														parser_waveform_count <= (others=>'0');
														ramUART_addr <= parser_waveform_count;
														parser_state <= W_counting;
													elsif uart_rxd = 16#3F# then --was '?'
														parser_state <= print_info;
													elsif uart_rxd = 16#4D# then --was 'M'
														parser_state <= M_getmode;
													end if;
													test_from_host <= "000";
													ramUART_we_n <= '1';
													
					when M_getmode =>
													case uart_rxd(3 downto 0) is --zero equals "00110000";
														when "0000" =>
															test_from_host <= "001";
														when "0001" =>
															test_from_host <= "010";
														when "0010" =>
															test_from_host <= "100";
														when others => null;
													end case;
													parser_state <= wait_for_F;
	
					when W_counting =>		
													ramUART_addr <= parser_waveform_count;
													ramUART_din <= uart_rxd;
													ramUART_we_n <= '0';
													
													if parser_waveform_count="11111111" then
														parser_state <= wait_for_F;
														should_switch_waveform_bank <= '1';
													else
														parser_waveform_count <= parser_waveform_count+1;
													end if;
	
					when F_before_decimal =>if uart_rxd = "00101110" then -- was a dot '.'
														parser_state <= F_after_decimal;
													elsif uart_rxd = "00001010" or uart_rxd = "00001101" then
														should_calculate_inc<='1';
														parser_state <= F_waiting_for_mul;
														--inc_temp := (parser_freq*top);--(32+22);
														--inc_pipelined <= inc_temp((32+16-1) downto 16);
													else
														-- parser_freq = 10*parser_freq + "uart_rxd as a number"
														-- "110000"=48=character '0'
														parser_freq <= ( parser_freq(28 downto 0) & "000" ) + (parser_freq(30 downto 0)&"0") + ("000000000000000000000000" & (uart_rxd - "00110000")); 
													end if;
	
	
					when F_after_decimal  =>	-- DUPLICATE this inside state before_decimal
--													if uart_rxd = "00001010" or uart_rxd = "00001101" then
														--parser_state <= F_waiting_for_mul;
--														--if resetn='1' then
--															inc_temp := (parser_freq*top);--(47 downto 16);
--															inc <= inc_temp(32+16-1 downto 16);
--															--inc <= inc(23 downto 16);
--															--inc <= CONV_STD_LOGIC_VECTOR((parser_freq*top)/bottom,24); -- ONLY VALID when oscillator freq=2^24Hz (approx 16.777MHz)
--														--end if;
--													end if;
	
					
	
					when on_newline  =>		parser_state <= wait_for_newline;
	
		
					when others => null;
					
				end case;
				uart_clear_rxd_full<='1';
				
				
			elsif uart_rxd_full='1' then
				uart_clear_rxd_full<='1';
			end if;
			
			uart_rxd_full_delay := uart_rxd_full;
			
			--end if;
			
		end if;
	end process;
end rtl;

