/**
 * @file gtkmm-related.hh
 * @todo Add reading of waveform from file (one number per line, as well as raw binary)
 * @author Simon Gustafsson (www.simong.se)
 **/

#ifndef GTKMMRELATED_HH_
#define GTKMMRELATED_HH_

#include <gtkmm.h>
#include <giomm.h>

#include <gtkmm/drawingarea.h> //for MyArea
#include <gtkmm/comboboxtext.h>
#include <gtkmm/statusbar.h>
#include <iostream>

#include "Serial.hh"

#define sqr(x) (x)*(x)

//==============================================================
enum IntersectType{
	NONE=0,
	OBJECT=1,
	BOTH_HANDLE=2,
	START_HANDLE=4,
	END_HANDLE=8,
};

//================================================================

struct point_t{
	int x;
	int y;
	point_t(int x_, int y_): x(x_), y(y_) {}
};



class MyArea : public Gtk::DrawingArea
{
public:

	MyArea();
	virtual ~MyArea();

	Gtk::TextView* textView; ///< We need this reference to highlight rows in editor...
	Gtk::Label* m_StatusPosition; ///<For output to the status bar...;
	std::vector<point_t> lines;
	
	int highlight_line;
	int mousex, mousey;

	void setTextView(Gtk::TextView* textView_){
		textView = textView_;
	}
	void setStatusLabel(Gtk::Label* m_StatusPosition_){
		m_StatusPosition = m_StatusPosition_;
	}
protected:
	//Override default signal handler:
	virtual bool on_expose_event(GdkEventExpose* event);
	
	virtual bool on_motion_notify_event(GdkEventMotion* event);
};

MyArea::MyArea(){
	textView=0;
	m_StatusPosition=0;
	//draw = DRAW_XY;
	highlight_line=-1;
	mousex=mousey=0;
	add_events(Gdk::POINTER_MOTION_MASK); //to get events into on_motion_notify_event
}
MyArea::~MyArea(){

}

bool MyArea::on_expose_event(GdkEventExpose* event)
{
	// This is where we draw on the window
	Glib::RefPtr<Gdk::Window> window = get_window();
	if(window)
	{
		Gtk::Allocation allocation = get_allocation();
		const int width = allocation.get_width();
		const int height = allocation.get_height();

//		global_trans->setParams(width, height, borders);

		// coordinates for the center of the window
		int xc, yc;
		xc = width / 2;
		yc = height / 2;
		Cairo::RefPtr<Cairo::Context> cr = window->create_cairo_context();
		cr->set_line_width(1.0);
		// clip to the area indicated by the expose event so that we only redraw
		// the portion of the window that needs to be redrawn
		cr->rectangle(event->area.x, event->area.y,
				event->area.width, event->area.height);
		cr->clip();
		// draw red lines out from the center of the window
		cr->set_source_rgb(0.8, 0.0, 0.0);
		
		
		unsigned char data[256*256*4];
		for(int y=0; y<256; y++){
			for(int x=0; x<256; x++){
				data[(y*256+x)*3 + 0] = ((y&16 && !(y&15)) || (x&16 && !(x&15)))?128:255;
				data[(y*256+x)*3 + 1] = ((y&16 && !(y&15)) || (x&16 && !(x&15)))?128:255;
				data[(y*256+x)*3 + 2] = ((y&16 && !(y&15)) || (x&16 && !(x&15)))?128:255;
				
			}
		}
		
		Glib::RefPtr<Gdk::GC> some_gc = Gdk::GC::create(window);

		Glib::RefPtr<Gdk::Pixbuf> image = Gdk::Pixbuf::create_from_data( (const guint8*)data,
		Gdk::COLORSPACE_RGB,
		0	/*has_alpha*/,
		8	/*bits_per_sample*/,
		256	/*width*/,
		256	/*height*/,
		256*3	/*rowstride*/);
		
		
		// Playing around with  pixbufs, to see if they can be used for animations. (should really check if one can iddle)
		//
		//http://sourceforge.net/projects/gtksdl/
		//
		//for(int i=0; i<256; i++){
			window->draw_pixbuf(some_gc, image,
			0,
			0,
			0,//i,
			0,
			256,
			256,
			Gdk::RGB_DITHER_NONE,
			0,
			0);
		//}
		
		
		for(int i=0; i<(lines.size()-1); i++){
			cr->move_to( lines[i].x, 255-lines[i].y );
			cr->line_to( lines[i+1].x, 255-lines[i+1].y );
			cr->stroke();
		}
		
		
	}
	return true;
}



bool MyArea::on_motion_notify_event(GdkEventMotion* event){
	//std::cout << "x,y=" <<event->x << "," << event->y << "\n";
	int redraw = (mousex != event->x ) || (mousey != event->y );
	mousex=event->x;
	mousey=event->y;
	
	Gtk::Allocation allocation = get_allocation();
	const int width = allocation.get_width();
	const int height = allocation.get_height();

	
	if(redraw)
		queue_draw();
	
	return true;
}


class Application : public Gtk::Window{
public:
	Application(const char* dev = "/dev/ttyUSB0");
protected:
	//Signal handlers
	virtual void on_button_clicked();
	virtual void on_button_file_clicked();
	virtual void on_viewcombo_changed();
	
	virtual void on_menu_file_new_generic();
	virtual void on_menu_file_quit();
	virtual void on_menu_others();
	virtual void on_menu_choices_sine();
	virtual void on_menu_choices_square();
	virtual void on_menu_choices_sawtooth();

	virtual void on_spinbutton_digits_changed();
	
	//Helper functions
	void sendWaveform(unsigned char* buffer, int size=255);
	void sendFrequency(int freq);
	void useWaveformRam();
	
	
	//Child Widgets
	Gtk::VBox m_vbox, m_vbox2;
	Gtk::HBox m_hbox1, m_hbox2;
	Gtk::HBox m_freq_and_set;
	Gtk::HButtonBox m_hbbox;
	Gtk::Statusbar m_Statusbar;
	Gtk::Label m_StatusPosition;

	Gtk::ScrolledWindow m_ScrolledWindow;
	Gtk::TextView m_TextView;
	//Glib::RefPtr<Gtk::TextBuffer> m_refTextBuffer;

	Gtk::HPaned m_HPaned;
	Gtk::Label m_Label_Normal;
	Gtk::Button m_Button_Close, m_Button_File;

	Gtk::ComboBoxText m_ViewCombo;

	Gtk::Adjustment m_adjustment_freq;
	Gtk::SpinButton m_SpinButton_freq;

	//my drawing area...
	MyArea area;

	//Classes sewing together toolbar and menu
	Glib::RefPtr<Gtk::UIManager> m_refUIManager;
	Glib::RefPtr<Gtk::ActionGroup> m_refActionGroup;
	
	//REMOVE THESE - JUST TESTING MENUS...
	Glib::RefPtr<Gtk::RadioAction> m_refChoiceOne, m_refChoiceTwo, m_refChoiceThree;
	
	//For connection to serial port.
	Serial serial;
	int fd;
};

Application::Application(const char* dev) : 
	//m_vbox(false,10), 
	//m_hbox1(false,10),
	//m_hbox2(false,10),
	//m_hbbox(),
	m_Label_Normal("test"),
	m_Button_Close(Gtk::Stock::QUIT), //,("close")
	m_Button_File(Gtk::Stock::OPEN),
	m_adjustment_freq(0, 0.0, 1000000.0, 1.0, 100.0, 0.0),
	m_SpinButton_freq(m_adjustment_freq),
	//m_Button_Quit(Gtk::Stock::QUIT)
	serial(dev)
	{
		fd = (int)serial;
		if(fd==0){
			perror("open");
			std::cout << "Error opening console \"" << dev << "\" for writing." << std::endl;
			//exit(2);
		}
		
	// Set title and border of the window
	set_title("VHDL-DDS");
	set_border_width(5);
	set_default_size(400, 200);

	//Add outer box to the window (because the window
	//can only contain a single widget)
	add(m_vbox);

	//Put the inner boxes and the separator in the outer box:
	m_vbox.pack_start(m_vbox2, Gtk::PACK_SHRINK);
	
	m_SpinButton_freq.set_numeric(1);
	m_vbox.pack_start(m_SpinButton_freq, Gtk::PACK_SHRINK);
	
	
	m_vbox.pack_start(m_HPaned);
	//m_vbox.pack_start(m_hbbox, Gtk::PACK_SHRINK);
	m_vbox.pack_start(m_Statusbar, Gtk::PACK_SHRINK);
	
	//add fixed widgets to the status bar
	//m_Statusbar.pack_end(m_StatusPosition);

	//Set the inner boxes’ borders
	//m_hbox1.set_border_width(10);
	m_HPaned.set_border_width(10);

	//Add the TextView, inside a ScrolledWindow
	m_ScrolledWindow.add(m_TextView);
	//Only show the scrollbars when they are necessary:
	m_ScrolledWindow.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);

	m_ScrolledWindow.set_size_request(200, 120);

	m_HPaned.add1(m_ScrolledWindow);
	area.set_size_request(256, 256);
	area.setTextView(&m_TextView);
	area.setStatusLabel(&m_StatusPosition);
	m_HPaned.pack2(area);
	area.show();

	// Put Close button in m_hbbox:
	//m_hbbox.pack_start(m_Button_Close, Gtk::PACK_SHRINK);
	//m_hbbox.pack_start(m_ViewCombo);
	//m_hbbox.pack_start(m_Button_File, Gtk::PACK_SHRINK);


	m_ViewCombo.append_text("Sine");
	m_ViewCombo.append_text("Square");
	m_ViewCombo.set_active_text("Sawtooth");


	// Put button box in m_hbox2:
	//m_hbox2.pack_start(m_hbbox, Gtk::PACK_SHRINK);

	// Make the button the default widget
	m_Button_Close.set_flags(Gtk::CAN_DEFAULT);
	m_Button_Close.grab_default();

	// Connect the clicked signal of the button to
	// RadioButtons::on_button_clicked()
	m_Button_Close.signal_clicked().connect(sigc::mem_fun(*this,
			&Application::on_button_clicked) );
	m_Button_File.signal_clicked().connect(sigc::mem_fun(*this,
			&Application::on_button_file_clicked) );

	m_ViewCombo.signal_changed().connect(sigc::mem_fun(*this,
			&Application::on_viewcombo_changed) );

	
	
	
	m_SpinButton_freq.signal_changed().connect(sigc::mem_fun(*this,
			&Application::on_spinbutton_digits_changed) );
	
	
	//Create actions for menus and toolbars:
	m_refActionGroup = Gtk::ActionGroup::create();
	
	//File|Open
	m_refActionGroup->add(Gtk::Action::create("FileOpen",
		            Gtk::Stock::OPEN, "_Open...", "Open an existing file"),
		            sigc::mem_fun(*this, &Application::on_button_file_clicked));
	//File|New sub menu:
	m_refActionGroup->add(Gtk::Action::create("FileNewStandard",
	            Gtk::Stock::NEW, "_New", "Create a new file"),
	        sigc::mem_fun(*this, &Application::on_menu_file_new_generic));
	m_refActionGroup->add(Gtk::Action::create("FileNewFoo",
	            Gtk::Stock::NEW, "New Foo", "Create a new foo"),
	        sigc::mem_fun(*this, &Application::on_menu_file_new_generic));
	m_refActionGroup->add(Gtk::Action::create("FileNewGoo",
	            Gtk::Stock::NEW, "_New Goo", "Create a new goo"),
	        sigc::mem_fun(*this, &Application::on_menu_file_new_generic));
	//File menu:
	m_refActionGroup->add(Gtk::Action::create("FileMenu", "File"));
	//Sub-menu.
	m_refActionGroup->add(Gtk::Action::create("FileNew", Gtk::Stock::NEW));
	m_refActionGroup->add(Gtk::Action::create("FileQuit", Gtk::Stock::QUIT),
	        sigc::mem_fun(*this, &Application::on_menu_file_quit));
	//Edit menu:
	m_refActionGroup->add(Gtk::Action::create("EditMenu", "Edit"));
	m_refActionGroup->add(Gtk::Action::create("EditCopy", Gtk::Stock::COPY),
	        sigc::mem_fun(*this, &Application::on_menu_others));
	m_refActionGroup->add(Gtk::Action::create("EditPaste", Gtk::Stock::PASTE),
	        sigc::mem_fun(*this, &Application::on_menu_others));
	m_refActionGroup->add(Gtk::Action::create("EditSomething", "Something"),
	        Gtk::AccelKey("<control><alt>S"),
	        sigc::mem_fun(*this, &Application::on_menu_others));
	
	//Choices menu, to demonstrate Radio items
	m_refActionGroup->add( Gtk::Action::create("WaveMenu", "Waveform") );
	Gtk::RadioAction::Group group_userlevel;
	m_refChoiceOne = Gtk::RadioAction::create(group_userlevel, "WaveSine", "Sine");
	m_refActionGroup->add(m_refChoiceOne,
	        sigc::mem_fun(*this, &Application::on_menu_choices_sine) );
	m_refChoiceTwo = Gtk::RadioAction::create(group_userlevel, "WaveSquare", "Square");
	m_refActionGroup->add(m_refChoiceTwo,
	        sigc::mem_fun(*this, &Application::on_menu_choices_square) );
	        
	m_refChoiceThree = Gtk::RadioAction::create(group_userlevel, "WaveSawtooth", "Sawtooth");
	m_refActionGroup->add(m_refChoiceThree,
	        sigc::mem_fun(*this, &Application::on_menu_choices_sawtooth) );
	        
	//Help menu:
	m_refActionGroup->add( Gtk::Action::create("HelpMenu", "Help") );
	m_refActionGroup->add( Gtk::Action::create("HelpAbout", Gtk::Stock::HELP),
	        sigc::mem_fun(*this, &Application::on_menu_others) );
	m_refUIManager = Gtk::UIManager::create();
	m_refUIManager->insert_action_group(m_refActionGroup);
	add_accel_group(m_refUIManager->get_accel_group());
	//Layout the actions in a menubar and toolbar:
	Glib::ustring ui_info =
		"<ui>"
		" <menubar name='MenuBar'>"
		"    <menu action='FileMenu'>"
		"      <menuitem action='FileOpen'/>"
		//"      <menu action='FileNew'>"
		//"        <menuitem action='FileNewStandard'/>"
		//"        <menuitem action='FileNewFoo'/>"
		//"        <menuitem action='FileNewGoo'/>"
		//"      </menu>"
		//"      <separator/>"
		"      <menuitem action='FileQuit'/>"
		"    </menu>"
		"    <menu action='WaveMenu'>"
		"      <menuitem action='WaveSine'/>"
		"      <menuitem action='WaveSquare'/>"
		"      <menuitem action='WaveSawtooth'/>"
		"    </menu>"
		//"    <menu action='EditMenu'>"
		//"      <menuitem action='EditCopy'/>"
		//"      <menuitem action='EditPaste'/>"
		//"      <menuitem action='EditSomething'/>"
		//"    </menu>"
		//"    <menu action='ChoicesMenu'>"
		//"      <menuitem action='ChoiceOne'/>"
		//"      <menuitem action='ChoiceTwo'/>"
		//"    </menu>"
		//"    <menu action='HelpMenu'>"
		//"      <menuitem action='HelpAbout'/>"
		//"    </menu>"
		" </menubar>"
		" <toolbar name='ToolBar'>"
		"    <toolitem action='FileOpen'/>"
		//"    <toolitem action='FileNewStandard'/>"
		"    <toolitem action='FileQuit'/>"
		" </toolbar>"
		"</ui>";
	#ifdef GLIBMM_EXCEPTIONS_ENABLED
	try
	{
	  m_refUIManager->add_ui_from_string(ui_info);
	}
	catch(const Glib::Error& ex)
	{
	  std::cerr << "building menus failed: " << ex.what();
	}
	#else
	std::auto_ptr<Glib::Error> ex;
	m_refUIManager->add_ui_from_string(ui_info, ex);
	if(ex.get())
	{
	  std::cerr << "building menus failed: " << ex->what();
	}
	#endif //GLIBMM_EXCEPTIONS_ENABLED
	//Get the menubar and toolbar widgets, and add them to a container widget:
	Gtk::Widget* pMenubar = m_refUIManager->get_widget("/MenuBar");
	if(pMenubar)
		m_vbox2.pack_start(*pMenubar, Gtk::PACK_SHRINK);
	Gtk::Widget* pToolbar = m_refUIManager->get_widget("/ToolBar") ;
	if(pToolbar)
		m_vbox2.pack_start(*pToolbar, Gtk::PACK_SHRINK);
	
	
	
	
	
	
	
	
	
	
	
	// Show all children of the window
	show_all_children();
	
	on_menu_choices_sine();
	useWaveformRam();

	}	

void Application::on_button_file_clicked(){
	Gtk::FileChooserDialog dialog("Please choose a file", Gtk::FILE_CHOOSER_ACTION_OPEN);

	dialog.set_transient_for(*this);
	//Add response buttons the the dialog:
	dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	dialog.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_OK);

	//Add filters, so that only certain file types can be selected:
	Gtk::FileFilter filter_text;
	filter_text.set_name("Text files");
	filter_text.add_mime_type("text/plain");
	dialog.add_filter(filter_text);
	Gtk::FileFilter filter_cpp;
	filter_cpp.set_name("C/C++ files");
	filter_cpp.add_mime_type("text/x-c");
	filter_cpp.add_mime_type("text/x-c++");
	filter_cpp.add_mime_type("text/x-c-header");
	dialog.add_filter(filter_cpp);
	Gtk::FileFilter filter_any;
	filter_any.set_name("Any files");
	filter_any.add_pattern("*");
	dialog.add_filter(filter_any);
	//Show the dialog and wait for a user response:
	int result = dialog.run();
	//Handle the response:
	switch(result)
	{
	case(Gtk::RESPONSE_OK):
	{
		std::cout << "Open clicked." << std::endl;
		//Notice that this is a std::string, not a Glib::ustring.
		std::string filename = dialog.get_filename();
		std::cout << "File selected: " << filename << std::endl;
		//area.setParser(new GCodeParser(filename.c_str()));

		Glib::RefPtr<Gtk::TextBuffer> tb = m_TextView.get_buffer();

		Glib::RefPtr<Gio::File>  myfile =	Gio::File::create_for_path(filename.c_str());
		//Glib::RefPtr<Gio::FileInputStream> input = myfile->read();
		char* contents;
		gsize length;
		std::string etag_out;
		myfile->load_contents (contents, length, etag_out); //Not nice if null chars in text...
		if(length>0){
			tb->set_text(contents, contents+length-1);
		}else{
			tb->set_text("");
		}
		area.queue_draw();


		break;
	}
	case(Gtk::RESPONSE_CANCEL):
	{
		std::cout << "Cancel clicked." << std::endl;
		break;
	}
	default:
	{
		std::cout << "Unexpected button clicked." << std::endl;
		break;
	}
	}
}


void Application::on_button_clicked(){
	hide(); //to close the application.
}

void Application::on_viewcombo_changed(){
	std::cout << __func__ << " ";

	Glib::ustring text = m_ViewCombo.get_active_text();

	if(text.compare("Show XY")==0){

		//delete global_trans;
		//global_trans = new XY_Transformation;
		//Path::setTransformation(global_trans);
		//area.draw = MyArea::DRAW_XY;

	}else if(text.compare("Show isom")==0){
		//delete global_trans;
		//global_trans = new Isom_Transformation;
		//Path::setTransformation(global_trans);
		//area.draw = MyArea::DRAW_ORTHO;
	}else{
		if(!(text.empty()))
			std::cout << "Combo changed: " << text << std::endl;
	}
	std::cout << "\n";
	area.queue_draw();
}





void Application::on_spinbutton_digits_changed()
{
  //m_SpinButton_freq.set_digits( m_SpinButton_freq.get_value_as_int() );
  std::cout << "Frequency set to " << m_SpinButton_freq.get_value_as_int() << "\n";
  sendFrequency( m_SpinButton_freq.get_value_as_int() );
}








void Application::on_menu_file_quit()
{
  hide(); //Closes the main window to stop the Gtk::Main::run().
}
void Application::on_menu_file_new_generic()
{
   std::cout << "A File|New menu item was selected." << std::endl;
}
void Application::on_menu_others()
{
  std::cout << "A menu item was selected." << std::endl;
}

void Application::sendWaveform(unsigned char* buffer, int size){
	char cmd='W';
	int res = write(fd, (void*)&cmd, 1);
	if(res!=1){
		perror("write 1 in sendWaveform");
		std::cout << __FILE__ << ":" << __LINE__ << "didn't send all bytes\n";
	}
	res = write(fd, (void*)buffer, size);
	if(res!=size){
		perror("write 2 in sendWaveform");
		std::cout << __FILE__ << ":" << __LINE__ << "didn't send all bytes\n";
	}
	cmd='\n';
	res = write(fd, (void*)&cmd, 1);
	if(res!=1){
		perror("write 3 in sendWaveform");
		std::cout << __FILE__ << ":" << __LINE__ << "didn't send all bytes\n";
	}
	
}

void Application::sendFrequency(int freq){
	
	char buff[20];
	snprintf(buff,sizeof(buff),"F%d\n\0",freq);
	
	int res = write(fd, (void*)buff, strlen(buff));
	
	if(res!=strlen(buff)){
		perror("write in sendFrequency");
		std::cout << __FILE__ << ":" << __LINE__ << "didn't send all bytes\n";
	}
}

void Application::useWaveformRam(){
	
	char buff[20];
	snprintf(buff,sizeof(buff),"M2\n\0");
	
	int res = write(fd, (void*)buff, strlen(buff));
	
	if(res!=strlen(buff)){
		perror("write in useWaveformRam");
		std::cout << __FILE__ << ":" << __LINE__ << "didn't send all bytes\n";
	}
}

void Application::on_menu_choices_sine()
{
  Glib::ustring message;
  if(m_refChoiceOne->get_active()){
    message = "Choice 1 was selected.";
    //global_trans = new XY_Transformation;
    //Path::setTransformation(global_trans);
    
    area.lines.clear();
    unsigned char data[256];
    for(int i=0; i<256; i++){
    	data[i] = round(127.5*(1+sin(i*2*M_PI/256)));
    	point_t tmp(i, data[i] );
    	area.lines.push_back(tmp);
	}
    sendWaveform(data,256);
    
    area.queue_draw();
  }else
    message = "Choice 1 was deselected";
  std::cout << message << std::endl;
}
void Application::on_menu_choices_square()
{
  Glib::ustring message;
  if(m_refChoiceTwo->get_active()){
    message = "Choice 2 was selected.";
    //global_trans = new Isom_Transformation;
    //Path::setTransformation(global_trans);
    
    area.lines.clear();
    unsigned char data[256];
    for(int i=0; i<256; i++){
    	data[i] = (i&128)?255:0;
    	point_t tmp(i, data[i]);
    	area.lines.push_back(tmp);
	}
	sendWaveform(data,256);
    
    
    area.queue_draw();
  }else
    message = "Choice 2 was deselected";
  std::cout << message << std::endl;
}

void Application::on_menu_choices_sawtooth()
{
  Glib::ustring message;
  if(m_refChoiceThree->get_active()){
    message = "Choice 3 was selected.";
    //global_trans = new Isom_Transformation;
    //Path::setTransformation(global_trans);
    
    area.lines.clear();
    unsigned char data[256];
    for(int i=0; i<256; i++){
    	data[i] = i;
    	point_t tmp(i, data[i]);
    	area.lines.push_back(tmp);
	}
	sendWaveform(data,256);
    
    
    area.queue_draw();
  }else
    message = "Choice 3 was deselected";
  std::cout << message << std::endl;
}

#endif /*GTKMMRELATED_HH_*/
