/* ****************************************************
Kod som tar emot kommandon seriellt, och PWM-modulerar tv motorer
**************************************************** */
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/signal.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h> 
#include <stdlib.h>
#include <inttypes.h>

/* Definitioner */
#define CLOCK 4000000

#define PPM_IN0		PD2
//PD2 = INT0, hr kommer en servostyrningssignal in

#define PPM_IN1		PD3
//PD3 = INT1, hr kommer andra servostyrningssignal in

#define SWITCH_INDIVIDUAL PD0
#define SWITCH_nDISABLE PD5
#define OUT_WDT_TOGGLE PD4

#define IS_INDIVIDUAL (bit_is_set(PIND,SWITCH_INDIVIDUAL))
#define IS_DISABLED (bit_is_clear(PIND,SWITCH_nDISABLE))
#define TOGGLE_OUT_WDT_TOGGLE ( PORTD=PORTD ^ _BV(OUT_WDT_TOGGLE) )

/***************************************************/
#define PRESCALER0 8
#define COUNT0_250U ( ( 256 - (CLOCK/4000)/PRESCALER0 ) )
#define COUNT0_125U ( ( 256 - (CLOCK/8000)/PRESCALER0 ) )
#define PRESCALER1 8
#define COUNT1_1ms ( (CLOCK/1000)/PRESCALER1  )

//Denna r delad med tta, eftersom rknarens vre delat med tta dr denna anvnds
#define COUNT1_1MS5_DIV_8 ( (CLOCK/667)/PRESCALER1/8 )

// dessa saker betyder inte min och max, utan anvnds till annat...
#define COUNT1_middle ( (CLOCK/667)/PRESCALER1 )
#define COUNT1_min ( (CLOCK/2000)/PRESCALER1  )
#define COUNT1_max ( (CLOCK/400)/PRESCALER1  )

#define COUNT1_50MS ( (CLOCK/20)/PRESCALER1 )

//Dessa saker hjlper en att switcha rtt saker.
//SW=switch, M0=motor0, M1=motor1, L/R = hger/vnster del av H-bryggan, 
//H/L = high/low p sista position i "makrot"
#define SW_M0_L_H PB7
#define SW_M0_L_L PB6
#define SW_M0_R_H PB5
#define SW_M0_R_L PB4

#define SW_M1_L_H PB3
#define SW_M1_L_L PB2
#define SW_M1_R_H PB1
#define SW_M1_R_L PB0

#define FET_OFF(x) (sbi(PORTB,x))
#define FET_ON(x)  (cbi(PORTB,x))

/* Globala variabler */
int8_t volatile dir0=0;
int8_t volatile dir1=0; //Rrelseriktning. 0=stilla, 1=frammt, -1=bakt...
uint8_t speed0, speed1; //Utrknad hastighet... 
int8_t volatile speedcnt=0; //Rknare fr att "mata ut hastighet".
uint16_t cntstart0, cntstart1, cnt0, cnt1; //Anvnds fr indatahantering
uint8_t volatile myflags=0; //d0=int0 skedde, d1=int1 skedde, d7=timeout i vntan p insignal

int8_t volatile locked_dir0=0;
int8_t volatile locked_dir1=0;
/* Vi kan inte anvnda dirX direkt, d det kan kortsluta h-bryggan, utan kopierar ver dessa vrden
 * till locked_dirX. Detta behver dock inte gras med speed0 eller speed1 */
 
/* Funktionsprototyper */
void delay(uint16_t max);
void init_ports(void);
void init_counters(void);

/* Interrupthanterare */
SIGNAL(SIG_OVERFLOW0) 			/* rknare0. (kommer hit var 250us) */ //eller 125US
{
	//TCNT0 = COUNT0_250U; //Starta om rknare TCNT0 genom att ladda rknaren...
	TCNT0 = COUNT0_125U;
	
	speedcnt++;
	if(speedcnt >= 32) {
		speedcnt = 0;

		locked_dir0=dir0;
		locked_dir1=dir1;

		FET_OFF(SW_M0_L_L);
		FET_OFF(SW_M0_R_L);
		FET_OFF(SW_M1_L_L);
		FET_OFF(SW_M1_R_L);
		/*FIXME stt eventuellt en frdrjning hr s att undre switcharna      *
		 *hinner stngas av helt (tar absolut inte mer n ngra mikrosekunder)  */
		switch(locked_dir0){
		case  1: FET_ON(SW_M0_L_H);  FET_OFF(SW_M0_R_H);break;
		case  0: FET_OFF(SW_M0_L_H); FET_OFF(SW_M0_R_H);break;
		case -1: FET_OFF(SW_M0_L_H); FET_ON(SW_M0_R_H); break;		
		default: /*Ska inte intrffa, FIXME: hantera... */ break;
		}
		switch(locked_dir1){
		case  1: FET_ON(SW_M1_L_H);  FET_OFF(SW_M1_R_H);break;
		case  0: FET_OFF(SW_M1_L_H); FET_OFF(SW_M1_R_H);break;
		case -1: FET_OFF(SW_M1_L_H); FET_ON(SW_M1_R_H); break;		
		default: /*Ska inte intrffa, FIXME: hantera... */ break;
		}
	}
	
	if(speedcnt > speed0){
		FET_OFF(SW_M0_L_L);
		FET_OFF(SW_M0_R_L);
	}
	else //dvs speedcnt0 <= speed0
	{
		switch(locked_dir0){
		case  1: FET_OFF(SW_M0_L_L); FET_ON(SW_M0_R_L); break;		
		case  0: FET_OFF(SW_M0_L_L); FET_OFF(SW_M0_R_L);break;
		case -1: FET_ON(SW_M0_L_L);  FET_OFF(SW_M0_R_L);break;
		}
	}

	if(speedcnt > speed1){
		FET_OFF(SW_M1_L_L);
		FET_OFF(SW_M1_R_L);
	} 
	else //dvs speedcnt1 <= speed1
	{
		switch(locked_dir1){
		case  1: FET_OFF(SW_M1_L_L); FET_ON(SW_M1_R_L); break;
		case  0: FET_OFF(SW_M1_L_L); FET_OFF(SW_M1_R_L);break;
		case -1: FET_ON(SW_M1_L_L);  FET_OFF(SW_M1_R_L);break;
		}
	}
}

/* rknare1, Hit kommer vi om insignaler saknas. Rknare1 anvnds 
   ocks fr att mta pulstider fr servosignalerna. */
SIGNAL(SIG_OVERFLOW1)			
{
	myflags= myflags | 0x80; //Detta gr att slingan i main fortstter nd...

/*	Har flyttat ut dessa tgrder i main, fr att slippa gra cnt0 och cnt1 "volatile",
	d kompilatorn med optimiseringar lyckas anvnda dom korrekt i alla fall
	(frutom just detta fallet, d man hade behvt ha dom "volatile")
	
	//Ta reda p om vi kr i individuellt eller "larvbandslge"
	if(IS_INDIVIDUAL){
		// Det rcker om en signal kommer fram,
		// FIXME stoppa den motor som inte fr signaler..
		if(!((myflags & 0x01)==0x01)) { dir0=0; }
		if(!((myflags & 0x02)==0x02)) { dir1=0; }
	}
	else {
		// Larvbandslge. Bgge signalerna behvs. 
		// Stoppar maskinen fr skerhets skull
		dir0=0;
		dir1=0;
	}
*/
}

uint16_t temp; //Temp som anvnds av interrupt0 och interrupt1
SIGNAL(SIG_INTERRUPT0) //den ena servostyrningssignalens avkodning mha timer1 och yttre flank-interrupt
{
	cli();
	if(bit_is_set(MCUCR,ISC00)) {
		//Kom hit p stigande flank
		cntstart0=TCNT1;
		cbi(MCUCR,ISC00); // Gr s detta interrupt trigas p fallande flank.
	}
	else
	{
		//Kom hit p fallande flank
		temp=TCNT1;
		if (!(myflags & 0x01)) {
			cnt0=temp-cntstart0;
			sbi(MCUCR,ISC00); //Gr s detta interrupt triggas p stigande flank
			myflags = myflags | 0x01;
		}
	}
	sei();
}
SIGNAL(SIG_INTERRUPT1) //den andra servostyrningssignalens avkodning mha timer1 och yttre flank-interrupt
{
	cli();
	if(bit_is_set(MCUCR,ISC10)) {
		//Kom hit p stigande flank
		cntstart1=TCNT1;
		cbi(MCUCR,ISC10); // Gr s detta interrupt trigas p fallande flank.
	}
	else
	{
		//Kom hit p fallande flank
		temp=TCNT1;
		if (!(myflags & 0x02)) {
			cnt1=temp-cntstart1;
			sbi(MCUCR,ISC10); //Gr s detta interrupt triggas p stigande flank
			myflags = myflags | 0x02;
		}
	}
	sei();
}

/*********************[ MAIN ]********************************/

int main(void) 
{
	uint8_t x;
	int8_t tempx,tempy,temp0,temp1;

	uint16_t cnt0_old=COUNT1_middle;
	uint16_t cnt1_old=COUNT1_middle;
	
	init_ports();
	init_counters();
	wdt_reset();
	wdt_enable(WDTO_120MS);
	sei();

forever:
	while( ( bit_is_set(PIND,PPM_IN0) | bit_is_set(PIND,PPM_IN1) ) & !((myflags & 0x80) == 0x80)){
		//Vnta p att bda sakerna r lga, eller timeout...
		//
	};
	wdt_reset();
	TOGGLE_OUT_WDT_TOGGLE;
	cli();	
	myflags=0;
	cnt0=cnt1=0;	

	//Nollstller rknaren. Eftersom interrupt sker p owerflow, och vi vill ha det efter
	//50MS stller vi in grejerna s hr...
	TCNT1= (65535 - COUNT1_50MS); 


	sei();
	//outp(2,TCCR1B);		//PRESCALER1=8, rknaren gr
	
	outp(0x0F,MCUCR); //Anger att INT0 och INT1 triggas p stigande flank
	outp(0xC0,GIFR);  //Nollstller dessa interrupt (s inte de anropas nsta rad)
	outp(0xC0,GIMSK); //slr p interrupt frn INT0 och INT1 
	
	while(myflags < 3) {
		//Vntar hr p att bde interrupt 1 och 2 blir klara
		//ELLER att owerflow sker	
	};
	outp(0x00, GIMSK); //Stnger av INT0 och INT1
	
	//HANTERAR OWERFLOW genom att fejka att inget owerflow skett, samtidigt som den ltsas att
	//insignalerna har sina "centrumvrden"
	if((myflags & 0x80) == 0x80){
		//Overflow skedde, gr ngot bra...
		//Ta reda p om vi kr i individuellt eller "larvbandslge"
		if(IS_INDIVIDUAL){
			// Det rcker om en signal kommer fram,
			// stoppa den motor som inte fr signaler (mjukstopp)...
			if(!((myflags & 0x01)==0x01)) { cnt0=COUNT1_middle; }
			if(!((myflags & 0x02)==0x02)) { cnt1=COUNT1_middle; }
		}
		else {
			// Larvbandslge. Bgge signalerna behvs. 
			// Stoppar maskinen fr skerhets skull (mjukstopp)
			cnt0=COUNT1_middle;
			cnt1=COUNT1_middle;
		}
		myflags=(myflags | 0x03) & 0x7F; //FEJKAR att vi tagit emot signal frn bda...
	}
	
	//Stller servona i ngon sorts normalposition
	//om strning gjort att tiderna ndrats lngt utanfr normala 1-2 ms...
	if( cnt0 > COUNT1_max ) { cnt0=COUNT1_middle; }
	if( cnt1 > COUNT1_max ) { cnt1=COUNT1_middle; }
	if( cnt0 < COUNT1_min ) { cnt0=COUNT1_middle; }
	if( cnt1 < COUNT1_min ) { cnt1=COUNT1_middle; }

	//Anvnder terkopling frn tidigare cnt-vrden fr att utjmna strningar en aning...
	//Motsvarande ekvation "cnt0 = 0.25*cnt0_ny + 0.75*cnt0_old"
	//ger en stigtid fr en impuls p c:a 8 genomgngar av denna slinga, eller 200ms om bda 
	//servoingngarna matas med servopulser...
	cnt0=(cnt0 >> 2) + (cnt0_old >> 1) + (cnt0_old >> 2);
	cnt1=(cnt1 >> 2) + (cnt1_old >> 1) + (cnt1_old >> 2);
	cnt0_old=cnt0;
	cnt1_old=cnt1;

	if(IS_DISABLED){
		//Om pinnen PD5 lg, s ska bgge motorerna genast stoppas...
		dir0=dir1=0;
	}
	else 
	{
		//Bearbeta datat, och gr ngot konstruktivt med dir och speed
		if(IS_INDIVIDUAL){
		
			//Motorerna styrs individuellt
			if(myflags & 0x01) { //Frsta motorn
				x=(cnt0 >> 3) & 0xFF; 
				//Dessa vrden gller vid 4 MHz: count=0.003*(1/4E6)/8/8;
				//Fr 3 ms r detta 187
				//Fr 1.5 ms r detta 94
				//Fr 1 ms r detta 62.5
				if(x> COUNT1_1MS5_DIV_8 + 5) {
					dir0 = 1;
					speed0 = x - (COUNT1_1MS5_DIV_8 + 5); 
					//speed ligger troligen mellan 0 och 30
					//speed0 = (speed0 >> 1) + 1;
					//speed ligger troligen mellan 1 och 16
					goto found0;
				}
				if(x < COUNT1_1MS5_DIV_8 - 5) {
					dir0 = -1;
					speed0 = (COUNT1_1MS5_DIV_8 - 5) - x;
					//speed0 = (speed0 >> 1) + 1;
					goto found0;
				}
				dir0=0;
				found0:
			}
			
			if(myflags & 0x02) { //Andra motorn
				x=(cnt1 >> 3) & 0xFF; 
				//Dessa vrden gller vid 4 MHz: count=0.003*4E6/8/8;
				//Fr 3 ms r detta 187
				//Fr 1.5 ms r detta 94
				//Fr 1 ms r detta 62.5
				if(x> COUNT1_1MS5_DIV_8 + 5) {
					dir1 = 1;
					speed1 = x - (COUNT1_1MS5_DIV_8 + 5);
					//speed1 = (speed1 >> 1) + 1;
					goto found1;
				}
				if(x < COUNT1_1MS5_DIV_8 - 5) {
					dir1 = -1;
					speed1 = (COUNT1_1MS5_DIV_8 - 5) -x;
					//speed1 = (speed1 >> 1) + 1;
					goto found1;
				}
				dir1=0;
				found1:
			}		
	
		}
		else //Ska kra bandvagnsstyrning om inget i vgen fr det...
		{
			
			
			if(myflags & 0x80) { goto efter_styrning; } //Om owerflow, sker ingen "bandvagnsstyrning"
			
			tempx=((cnt0 >> 3) & 0xFF)-COUNT1_1MS5_DIV_8;
			tempy=((cnt1 >> 3) & 0xFF)-COUNT1_1MS5_DIV_8;
			//Dessa vrden gller vid 4 MHz: count=(0.003-0.0015)*(4E6)/8/8;
			//Fr 3 ms r detta 94
			//Fr 2 ms r detta 31
			//Fr 1.5 ms r detta 0
			//Fr 1 ms r detta -31
	
			//Ingen fart frammt - svng alltid p stllet.
			if(tempx>5) 
				{tempx=tempx-5;}
			else{
				if(tempx<-5) {tempx=tempx+5;}
				else { tempx=0;} //mellan -5 och 5
			}
			
			//Ingen svng --> g rakt fram
			if(tempy>5) {tempy=tempy-5;}
			else{
				if(tempy<-5) {tempy=tempy+5;}
				else {tempy=0;} //mellan -5 och 5
			}
			
			// tempx: -26 vid 1ms, +26 vid 2ms
			// tempy: -26 vid 1ms, +26 vid 2ms
			
			
			//temp0=(tempx >> 2) + (tempy >> 3);
			//temp1=(tempx >> 2) - (tempy >> 3);
			temp0=( (( tempy*(31-abs(tempx))) >> 5) + tempx);
			temp1=( ((-tempy*(31-abs(tempx))) >> 5) + tempx);
			
			
			//FIXME skala om storleken p dessa, s att speedX <=16...
			if (temp0>0) {dir0=1; speed0=temp0;}
			else {dir0=-1; speed0=-temp0;}
			if (temp0==0) {dir0=0;}
			
			
			//OBS! Hr stter vi dir omvnt mot tidigare, d motorerna
			//r monterade s att axlarna pekar t motsatta hll...
			//FIXME skala om storleken p dessa, s att speedX <=16...
			if (temp1>0) {dir1=-1; speed1=temp1;}
			else {dir1=1; speed1=-temp1;}
			if (temp1==0) {dir1=0;}
			
			//Motorerna styrs p larvbandsvis.
			//PPM_IN0 anger hastighet frammt-bakt, och
			//PPM_IN1 anger skilnad i hastighet mellan hger och vnster hjul..
	efter_styrning:		
			
		}
	}	
	goto forever;

	
}
/********************[SMSAKER]*******************************/
void init_ports(void){
	//Fixar riktning p portar.
	outp(0xFF,PORTB);
	outp(0xFF,DDRB); //Hela port B utgng
	
	outp(0x00 + _BV(OUT_WDT_TOGGLE),DDRD); //hela D ingng utom pinnen OUT_WDT_TOGGLE
	outp(0xFF,PORTD); //hela D pull-upp (utom pinnen OUT_WDT_TOGGLE, som r utgng)
}

void init_counters(void){
	//Rknare 0 ger interrupt var 250us eller 125 us mha owerflow/laddning av 
	//rtt vrde, och anvnds fr att pulsbreddsmodulera
	outp(0,TCNT0);		//Nollstll rknaren 
	outp(2,TCCR0);		//PRESCALER0=8 (FIXME), rknaren gr
	outp(COUNT0_125U,TCNT0);//Interrupt om 250us eller 125us
	
	//Rknare 1 ger endast owerflow-interrupt, och anvnds fr att 
	//tillsammans med INT0 och INT1 lsa in insignalerna...
	outp(0,TCCR1A);		//Nollstller vissa flaggor
	outp(0,TCCR1B);		//rknaren stoppad
	outp(0,TCNT1H);		//Nollstller rknaren
	outp(0,TCNT1L);		//Nollstller rknaren
	outp(2,TCCR1B);		//PRESCALER1=8, rknaren gr
	

	//Slr p TCNT0 och TCNT1 overflow-int
	outp((1<<TOIE0)|(1<<TOIE1),TIMSK);
	
}

void delay(uint16_t max)
{
/*	Ngra uppmtta tider vid 3.6864 MHz:
	max=100,  exekveringstid=193.41  us
	   =1000,               =1902.40 us
	   =10000,              =34.79 ms
	   
	OBS! (FIXME) detta kan kanske ndra sig 
	beroende p optimeringsinstllningar i GCC...
*/
	uint16_t i;
	for(i=0;i< max; i++) {
		asm volatile("nop\n\t"::);
	}
}
