#ifndef MyV4L2_HH
#define MyV4L2_HH

/*
 *  V4L2 video capture class
 *
 *  Copyright (c) 2007 Simon Gustafsson (www.simong.se)
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <getopt.h>             /* getopt_long() */

#include <fcntl.h>              /* low-level i/o */
#include <unistd.h>
#include <errno.h>
#include <malloc.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/ioctl.h>

#include <asm/types.h>          /* for videodev2.h */

#include <linux/videodev2.h>

#define CLEAR(x) memset (&(x), 0, sizeof (x))

/** Class to simplify use of the V4L2 interface for video capture.
    Completely based on information provided in
    "Video for Linux Two API Specification: Revision 0.23", which was located at <a href="http://www.linuxtv.org/downloads/video4linux/API/V4L2_API/spec/book1.htm">http://www.linuxtv.org/downloads/video4linux/API/V4L2_API/spec/book1.htm</a>

    (My personal local copy of it resides
    <a href="../../Desktop/V4L2%20API%20Specification%20(revision%200.23)/v4l2.html">somewhere on my desktop</a>.)

    @todo add posibility to turn automatic white balance on and off.

    @author Simon Gustafsson (www.simong.se)
*/
class MyV4L2{
public:

  /** Method used for IO */
  typedef enum {
    IO_METHOD_READ,
    IO_METHOD_MMAP,
    IO_METHOD_USERPTR,
  } io_method;

  struct buffer {
    void *                  start;
    size_t                  length;
  };

  /** V4L2 device name */
  char *           dev_name;
  /** Method used for IO */
  io_method        io;
  /** File descriptor (initialized to point to V4L2 device) */
  int              fd;
  /** Buffers for captured video */
  buffer *         buffers;
  /** Number of buffers used */
  unsigned int     n_buffers;

  /** Used pixel format (fourcc) */
  __u32 pixelformat;


  /**
     Constructor
     @param device devicename for V4L2-device
     @param iomethod requested method of io to device

     @param fourcc Requested pixel format in output buffers
@todo fourcc might be changed by driver not supporting requested fourcc. This might be good to relay to class user.
  */
  MyV4L2(char* device, io_method iomethod=IO_METHOD_READ, __u32 fourcc=V4L2_PIX_FMT_YUYV){

    if(device==0){
      dev_name=0;
    }else{
      int n = strlen(device)+1;
      dev_name = new char[n];
      strncpy(dev_name,device,n);
    }
    
    io=iomethod;

    fd=-1;
    buffers=NULL;
    n_buffers=0;

    pixelformat=fourcc;

  }

void
  errno_exit                      (const char* s)
  {
    fprintf (stderr, "%s error %d, %s\n",
	     s, errno, strerror (errno));

    exit (EXIT_FAILURE);
  }

int
  xioctl                          (int                    fd,
				   int                    request,
				   void *                 arg)
  {
    int r;

    do r = ioctl (fd, request, arg);
    while (-1 == r && EINTR == errno);

    return r;
  }

void process_image (const void* p)
  {
    //   fputc ('.', stdout);
    //    fflush (stdout);
  }

  /** @returns pointer to current video buffer, or 0 on error */
void* read_frame (void)
  {
    struct v4l2_buffer buf;
    unsigned int i;

    switch (io) {
    case IO_METHOD_READ:
      if (-1 == read (fd, buffers[0].start, buffers[0].length)) {
	switch (errno) {
	case EAGAIN:
	  return 0;

	case EIO:
	  /* Could ignore EIO, see spec. */

	  /* fall through */

	default:
	  errno_exit ("read");
	}
      }

      process_image (buffers[0].start);
      return buffers[0].start;
      break;

    case IO_METHOD_MMAP:
      CLEAR (buf);

      buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
      buf.memory = V4L2_MEMORY_MMAP;

      if (-1 == xioctl (fd, VIDIOC_DQBUF, &buf)) {
	switch (errno) {
	case EAGAIN:
	  return 0;

	case EIO:
	  /* Could ignore EIO, see spec. */

	  /* fall through */

	default:
	  errno_exit ("VIDIOC_DQBUF");
	}
      }

      assert (buf.index < n_buffers);

      process_image (buffers[buf.index].start);

      if (-1 == xioctl (fd, VIDIOC_QBUF, &buf))
	errno_exit ("VIDIOC_QBUF");

      return buffers[buf.index].start;
      break;

    case IO_METHOD_USERPTR:
      CLEAR (buf);

      buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
      buf.memory = V4L2_MEMORY_USERPTR;

      if (-1 == xioctl (fd, VIDIOC_DQBUF, &buf)) {
	switch (errno) {
	case EAGAIN:
	  return 0;

	case EIO:
	  /* Could ignore EIO, see spec. */

	  /* fall through */

	default:
	  errno_exit ("VIDIOC_DQBUF");
	}
      }

      for (i = 0; i < n_buffers; ++i)
	if (buf.m.userptr == (unsigned long) buffers[i].start
	    && buf.length == buffers[i].length)
	  break;

      assert (i < n_buffers);

      process_image ((void *) buf.m.userptr);

      if (-1 == xioctl (fd, VIDIOC_QBUF, &buf))
	errno_exit ("VIDIOC_QBUF");

      return (void*) buf.m.userptr;
      break;
    }

    return 0;
  }
 /** Read frame when it becomes available, or times out after five seconds.
     While waiting for the next frame, this process is sleeping.
     @returns pointer to current video buffer, or 0 on error */
  void* wait_read_frame()
  {
    for (;;) {
      fd_set fds;
      struct timeval tv;
      int r;
      
      //Watch fd to see when it has input.
      FD_ZERO (&fds);
      FD_SET (fd, &fds);
      
      // Timeout set to two seconds
      tv.tv_sec = 5;
      tv.tv_usec = 0;
      
      r = select (fd + 1, &fds, NULL, NULL, &tv);

      if (-1 == r) {
	if (EINTR == errno)
	  continue;

	errno_exit ("select");
      }

      if (0 == r) {
	fprintf (stderr, "select timeout\n");
	exit (EXIT_FAILURE);
      }

      void* retval;
      if ((retval=read_frame())!=0)
	return retval;
        
      /* EAGAIN - continue select loop. */
    }
    return 0;
  }

void
  mainloop                        (void)
  {
    unsigned int count;

    count = 100;

    while (count-- > 0) {
      for (;;) {
	fd_set fds;
	struct timeval tv;
	int r;

	FD_ZERO (&fds);
	FD_SET (fd, &fds);

	/* Timeout. */
	tv.tv_sec = 2;
	tv.tv_usec = 0;

	r = select (fd + 1, &fds, NULL, NULL, &tv);

	if (-1 == r) {
	  if (EINTR == errno)
	    continue;

	  errno_exit ("select");
	}

	if (0 == r) {
	  fprintf (stderr, "select timeout\n");
	  exit (EXIT_FAILURE);
	}

	if (read_frame ())
	  break;
        
	/* EAGAIN - continue select loop. */
      }
    }
  }

  /** Stop capturing. */
void
  stop_capturing                  (void)
  {
    enum v4l2_buf_type type;

    switch (io) {
    case IO_METHOD_READ:
      /* Nothing to do. */
      break;

    case IO_METHOD_MMAP:
    case IO_METHOD_USERPTR:
      type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

      if (-1 == xioctl (fd, VIDIOC_STREAMOFF, &type))
	errno_exit ("VIDIOC_STREAMOFF");

      break;
    }
  }

  /** Starts capturing.
      @note before capturing is started, ensure that 
      open_device(), set_input(), and init_device() has been called. 
      set_input() could be skipped, as the device remembers its last setting.
  */
void
  start_capturing                 (void)
  {
    unsigned int i;
    enum v4l2_buf_type type;

    switch (io) {
    case IO_METHOD_READ:
      /* Nothing to do. */
      break;

    case IO_METHOD_MMAP:
      for (i = 0; i < n_buffers; ++i) {
	struct v4l2_buffer buf;

	CLEAR (buf);

	buf.type        = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	buf.memory      = V4L2_MEMORY_MMAP;
	buf.index       = i;

	if (-1 == xioctl (fd, VIDIOC_QBUF, &buf))
	  errno_exit ("VIDIOC_QBUF");
      }
                
      type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

      if (-1 == xioctl (fd, VIDIOC_STREAMON, &type))
	errno_exit ("VIDIOC_STREAMON");

      break;

    case IO_METHOD_USERPTR:
      for (i = 0; i < n_buffers; ++i) {
	struct v4l2_buffer buf;

	CLEAR (buf);

	buf.type        = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	buf.memory      = V4L2_MEMORY_USERPTR;
	buf.index       = i;
	buf.m.userptr   = (unsigned long) buffers[i].start;
	buf.length      = buffers[i].length;

	if (-1 == xioctl (fd, VIDIOC_QBUF, &buf))
	  errno_exit ("VIDIOC_QBUF");
      }

      type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

      if (-1 == xioctl (fd, VIDIOC_STREAMON, &type))
	errno_exit ("VIDIOC_STREAMON");

      break;
    }
  }

  /** Uninits the device.
      Frees allocated buffers, and unmaps any mmap's
   */
void
  uninit_device                   (void)
  {
    unsigned int i;

    switch (io) {
    case IO_METHOD_READ:
      free (buffers[0].start);
      break;

    case IO_METHOD_MMAP:
      for (i = 0; i < n_buffers; ++i)
	if (-1 == munmap (buffers[i].start, buffers[i].length))
	  errno_exit ("munmap");
      break;

    case IO_METHOD_USERPTR:
      for (i = 0; i < n_buffers; ++i)
	free (buffers[i].start);
      break;
    }

    free (buffers);
  }

void
  init_read                       (unsigned int           buffer_size)
  {
    buffers = (buffer *) calloc (1, sizeof (*buffers));

    if (!buffers) {
      fprintf (stderr, "Out of memory\n");
      exit (EXIT_FAILURE);
    }

    buffers[0].length = buffer_size;
    buffers[0].start = malloc (buffer_size);

    if (!buffers[0].start) {
      fprintf (stderr, "Out of memory\n");
      exit (EXIT_FAILURE);
    }
  }

void
  init_mmap                       (void)
  {
    struct v4l2_requestbuffers req;

    CLEAR (req);

    req.count               = 4;
    req.type                = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    req.memory              = V4L2_MEMORY_MMAP;

    if (-1 == xioctl (fd, VIDIOC_REQBUFS, &req)) {
      if (EINVAL == errno) {
	fprintf (stderr, "%s does not support "
		 "memory mapping\n", dev_name);
	exit (EXIT_FAILURE);
      } else {
	errno_exit ("VIDIOC_REQBUFS");
      }
    }

    if (req.count < 2) {
      fprintf (stderr, "Insufficient buffer memory on %s\n",
	       dev_name);
      exit (EXIT_FAILURE);
    }

    buffers = (buffer *) calloc (req.count, sizeof (*buffers));

    if (!buffers) {
      fprintf (stderr, "Out of memory\n");
      exit (EXIT_FAILURE);
    }

    for (n_buffers = 0; n_buffers < req.count; ++n_buffers) {
      struct v4l2_buffer buf;

      CLEAR (buf);

      buf.type        = V4L2_BUF_TYPE_VIDEO_CAPTURE;
      buf.memory      = V4L2_MEMORY_MMAP;
      buf.index       = n_buffers;

      if (-1 == xioctl (fd, VIDIOC_QUERYBUF, &buf))
	errno_exit ("VIDIOC_QUERYBUF");

      buffers[n_buffers].length = buf.length;
      buffers[n_buffers].start =
	mmap (NULL /* start anywhere */,
	      buf.length,
	      PROT_READ | PROT_WRITE /* required */,
	      MAP_SHARED /* recommended */,
	      fd, buf.m.offset);

      if (MAP_FAILED == buffers[n_buffers].start)
	errno_exit ("mmap");
    }
  }

void
  init_userp                      (unsigned int           buffer_size)
  {
    struct v4l2_requestbuffers req;
    unsigned int page_size;

    page_size = getpagesize ();
    buffer_size = (buffer_size + page_size - 1) & ~(page_size - 1);

    CLEAR (req);

    req.count               = 4;
    req.type                = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    req.memory              = V4L2_MEMORY_USERPTR;

    if (-1 == xioctl (fd, VIDIOC_REQBUFS, &req)) {
      if (EINVAL == errno) {
	fprintf (stderr, "%s does not support "
		 "user pointer i/o\n", dev_name);
	exit (EXIT_FAILURE);
      } else {
	errno_exit ("VIDIOC_REQBUFS");
      }
    }

    buffers = (buffer *) calloc (4, sizeof (*buffers));

    if (!buffers) {
      fprintf (stderr, "Out of memory\n");
      exit (EXIT_FAILURE);
    }

    for (n_buffers = 0; n_buffers < 4; ++n_buffers) {
      buffers[n_buffers].length = buffer_size;
      buffers[n_buffers].start = memalign (/* boundary */ page_size,
					   buffer_size);

      if (!buffers[n_buffers].start) {
	fprintf (stderr, "Out of memory\n");
	exit (EXIT_FAILURE);
      }
    }
  }

void
  init_device                     (void)
  {
    struct v4l2_capability cap;
    struct v4l2_cropcap cropcap;
    struct v4l2_crop crop;
    struct v4l2_format fmt;
    unsigned int min;

    if (-1 == xioctl (fd, VIDIOC_QUERYCAP, &cap)) {
      if (EINVAL == errno) {
	fprintf (stderr, "%s is no V4L2 device\n",
		 dev_name);
	exit (EXIT_FAILURE);
      } else {
	errno_exit ("VIDIOC_QUERYCAP");
      }
    }

    if (!(cap.capabilities & V4L2_CAP_VIDEO_CAPTURE)) {
      fprintf (stderr, "%s is no video capture device\n",
	       dev_name);
      exit (EXIT_FAILURE);
    }

    switch (io) {
    case IO_METHOD_READ:
      if (!(cap.capabilities & V4L2_CAP_READWRITE)) {
	fprintf (stderr, "%s does not support read i/o\n",
		 dev_name);
	exit (EXIT_FAILURE);
      }

      break;

    case IO_METHOD_MMAP:
    case IO_METHOD_USERPTR:
      if (!(cap.capabilities & V4L2_CAP_STREAMING)) {
	fprintf (stderr, "%s does not support streaming i/o\n",
		 dev_name);
	exit (EXIT_FAILURE);
      }

      break;
    }


    /* Select video input, video standard and tune here. */


    CLEAR (cropcap);

    cropcap.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

    if (0 == xioctl (fd, VIDIOC_CROPCAP, &cropcap)) {
      crop.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
      crop.c = cropcap.defrect; /* reset to default */

      if (-1 == xioctl (fd, VIDIOC_S_CROP, &crop)) {
	switch (errno) {
	case EINVAL:
	  /* Cropping not supported. */
	  break;
	default:
	  /* Errors ignored. */
	  break;
	}
      }
    } else {        
      /* Errors ignored. */
    }


    CLEAR (fmt);

    fmt.type                = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    fmt.fmt.pix.width       = 640; 
    fmt.fmt.pix.height      = 480;
    fmt.fmt.pix.pixelformat = this->pixelformat;
    fmt.fmt.pix.field       = V4L2_FIELD_INTERLACED;

    if (-1 == xioctl (fd, VIDIOC_S_FMT, &fmt))
      errno_exit ("VIDIOC_S_FMT");

    /* Note VIDIOC_S_FMT may change width and height. */



    if(fmt.fmt.pix.pixelformat != this->pixelformat){
      printf("UNABLE TO SET REQUESTED PIXELFORMAT!\n");
      exit(-1);
    }


    /* Buggy driver paranoia. */
    min = fmt.fmt.pix.width * 2;
    if (fmt.fmt.pix.bytesperline < min)
      fmt.fmt.pix.bytesperline = min;
    min = fmt.fmt.pix.bytesperline * fmt.fmt.pix.height;
    if (fmt.fmt.pix.sizeimage < min)
      fmt.fmt.pix.sizeimage = min;

    switch (io) {
    case IO_METHOD_READ:
      init_read (fmt.fmt.pix.sizeimage);
      break;

    case IO_METHOD_MMAP:
      init_mmap ();
      break;

    case IO_METHOD_USERPTR:
      init_userp (fmt.fmt.pix.sizeimage);
      break;
    }
  }

  /** Closes the device */
void
  close_device                    (void)
  {
    if (-1 == close (fd))
      errno_exit ("close");

    fd = -1;
  }


  /** Opens device specified in the constructor of this class */
void
  open_device                     (void)
  {
    struct stat st; 

    if (-1 == stat (dev_name, &st)) {
      fprintf (stderr, "Cannot identify '%s': %d, %s\n",
	       dev_name, errno, strerror (errno));
      exit (EXIT_FAILURE);
    }

    if (!S_ISCHR (st.st_mode)) {
      fprintf (stderr, "%s is no device\n", dev_name);
      exit (EXIT_FAILURE);
    }

    fd = open (dev_name, O_RDWR /* required */ | O_NONBLOCK, 0);

    if (-1 == fd) {
      fprintf (stderr, "Cannot open '%s': %d, %s\n",
	       dev_name, errno, strerror (errno));
      exit (EXIT_FAILURE);
    }
  }

#if 0
  static void
  usage                           (FILE *                 fp,
				   int                    argc,
				   char **                argv)
  {
    fprintf (fp,
	     "Usage: %s [options]\n\n"
	     "Options:\n"
	     "-d | --device name   Video device name [/dev/video]\n"
	     "-h | --help          Print this message\n"
	     "-m | --mmap          Use memory mapped buffers\n"
	     "-r | --read          Use read() calls\n"
	     "-u | --userp         Use application allocated buffers\n"
	     "",
	     argv[0]);
  }

  static const char short_options [] = "d:hmru";

  static const struct option
  long_options [] = {
    { "device",     required_argument,      NULL,           'd' },
    { "help",       no_argument,            NULL,           'h' },
    { "mmap",       no_argument,            NULL,           'm' },
    { "read",       no_argument,            NULL,           'r' },
    { "userp",      no_argument,            NULL,           'u' },
    { 0, 0, 0, 0 }
  };

  int
  main                            (int                    argc,
				   char **                argv)
  {
    dev_name = "/dev/video";

    for (;;) {
      int index;
      int c;
                
      c = getopt_long (argc, argv,
		       short_options, long_options,
		       &index);

      if (-1 == c)
	break;

      switch (c) {
      case 0: /* getopt_long() flag */
	break;

      case 'd':
	dev_name = optarg;
	break;

      case 'h':
	usage (stdout, argc, argv);
	exit (EXIT_SUCCESS);

      case 'm':
	io = IO_METHOD_MMAP;
	break;

      case 'r':
	io = IO_METHOD_READ;
	break;

      case 'u':
	io = IO_METHOD_USERPTR;
	break;

      default:
	usage (stderr, argc, argv);
	exit (EXIT_FAILURE);
      }
    }

    open_device ();

    init_device ();

    start_capturing ();

    mainloop ();

    stop_capturing ();

    uninit_device ();

    close_device ();

    exit (EXIT_SUCCESS);

    return 0;
  }
#endif

 
    /**
       from http://www.linuxtv.org/downloads/video4linux/API/V4L2_API/spec/x303.htm:
Example 1-1. Information about the current video input
     */
    void show_input(){
      struct v4l2_input input;
      int index;
      
      if (-1 == ioctl (fd, VIDIOC_G_INPUT, &index)) {
	perror ("VIDIOC_G_INPUT");
	exit (EXIT_FAILURE);
      }
      
      memset (&input, 0, sizeof (input));
      input.index = index;
      
      if (-1 == ioctl (fd, VIDIOC_ENUMINPUT, &input)) {
	perror ("VIDIOC_ENUMINPUT");
	exit (EXIT_FAILURE);
      }
    
      printf ("Current input: %s\n", input.name);
    }

  void show_inputs(){
    printf("Inputs: ");
    struct v4l2_input input;
    for(int index=0; ; index++){
      memset (&input, 0, sizeof (input));
      input.index = index;

      if (-1 == ioctl (fd, VIDIOC_ENUMINPUT, &input)) {
	if(errno==EINVAL){
	  printf("\n");
	  return;
	}else{
	  perror ("VIDIOC_ENUMINPUT");
	  exit (EXIT_FAILURE);
	}
      }      
      printf (" %d = %s ", index, input.name);
    }
    printf("\n");
  }

  /**
     from http://www.linuxtv.org/downloads/video4linux/API/V4L2_API/spec/x303.htm:
     Example 1-2. Switching to the first video input
   */
  void set_input(int index=0){
    if (-1 == ioctl (fd, VIDIOC_S_INPUT, &index)) {
      perror ("VIDIOC_S_INPUT");
      exit (EXIT_FAILURE);
    }
  }


  void show_enumerate_image_formats(){
    struct v4l2_fmtdesc fmt;
    printf("Enumerated image formats:\n");
    for(__u32 index=0; ; index++){
      memset (&fmt, 0, sizeof (fmt));
      fmt.index = index;
      fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
      if(-1 == ioctl (fd, VIDIOC_ENUM_FMT, &fmt)){
	if(index==0){
	  perror ("VIDIOC_ENUM_FMT");
	}
	return;
      }
      printf(" %3d: %c%c%c%c - %.32s \n",index,

	     //v4l2_fourcc('G','R','E','Y') would set fourcc according to
	     //V4L2_PIX_FMT_GREY ('GREY'), and so on...
	     ((char*)&fmt.pixelformat)[0],
	     ((char*)&fmt.pixelformat)[1],
	     ((char*)&fmt.pixelformat)[2],
	     ((char*)&fmt.pixelformat)[3],
	     fmt.description //FIXME should be zero-terminated, don't want to assume that...
	     );
    }
  }

};

#endif
