#ifndef MFILEWRITER_HH
#define MFILEWRITER_HH

#include <fstream>
#include <vector>
#include <map>
#include <list>
#include <cstdlib>
#include <iostream>

/**
   Used for cashing up values to write as vectors in m-files.
   m-files are files containing matlab (or octave) data and commands.
   The write to the file is done at destruction of this class.
   It's possible to add any number of strings below the data,
   to perform additional operations.

   For those using octave, the installation of octave-forge is recommended,
   as this adds many additional commands. This class does not depend on any
   additional commands, but users of this class sometimes use advanced
   functions not yet included in the standard octave distribution.


@code
//Example MFileWriter usage

MFileWriter m("results.m",
              "%example of squaring a number, where xx(i)=x(i)*x(i)");
for(int i=1; i<4; i++){
  m.addToVector("x",i);
  m.addToVector("xx",i*i);
}
m.appendString("deviations=xx-x.*x;");
m.appendString("max_abs_deviations=max(abs(deviations))");
m.appendString("plot(x, xx, \";computer;\",x, x.*x, \";octave;\")");
@endcode

The code above would generate the following m-file:
@code
%Example of squaring a number, where xx(i)=x(i)*x(i)

%=====================[start of data]=========================

x = [1;
2;
3];

xx = [1;
4;
9];

%====[end of data (MFileReader won't read past this line)]====

%======================[Appended commands]====================

deviations=xx-x.*x;
max_abs_deviations=max(abs(deviations))
plot(x, xx, ";from computer;", x, x.*x, ";according to octave;")

%===============[end of appended commands]====================
@endcode


   @author Simon Gustafsson (www.simong.se)
*/
class MFileWriter{
 public:
  MFileWriter(std::string filename, std::string Heading="%MFileWriter generated file..."): _filename(filename), dest(_filename.c_str()){
    
    if(!dest) { 
      std::cerr << "Unable to open " << _filename << " for writing..." << std::endl; 
      /**@todo FIXME throw some kind of exception???*/
    }else {
      dest << Heading << std::endl << std::endl;
      dest.precision(10); //Keep enough decimals...
    }
    
  };
  ~MFileWriter(){
    std::cerr << "MFileWriter are storing data to \"" << _filename 
	      <<"\". This might take some time..." << std::endl;
    dest <<   "%=====================[start of data]=========================" 
	 << std::endl << std::endl;

    typedef std::map<std::string, std::vector<double> >::const_iterator CI;
    int numarrays=0;
    for(CI p=name_to_doublevector.begin(); p!=name_to_doublevector.end(); p++){
      numarrays++;
      dest << p->first << " = [";

      for(int i=0; i<((int)p->second.size()-1); i++){
	dest << p->second[i] << ";" << std::endl;
      }
      dest << p->second[p->second.size()-1] << "];" << std::endl << std::endl;
    }
    
    dest <<   "%====[end of data (MFileReader won't read past this line)]====";
    if(!appended_commands.empty()){
      dest << std::endl << std::endl
	   << "%======================[Appended commands]====================";
    }
    dest << std::endl << std::endl;

    typedef std::list<std::string>::const_iterator CI2;
    int numstrings=0;
    for(CI2 p=appended_commands.begin(); p!=appended_commands.end(); p++){
      numstrings++;
      dest << *p << std::endl;
    }
    if(!appended_commands.empty()){
      dest << std::endl
	   << "%===============[end of appended commands]===================="
	   << std::endl;
    }

    if(!dest){
      std::cerr << "MFileWriter couldn't sucessfully write to \"" 
		<< _filename << "\"." << std::endl;
    }else{
      std::cout << "MFileWriter sucessfully wrote " << numarrays 
		<< " vectors and " << numstrings << " additional commands "
		<< "to \""  << _filename << "\"." << std::endl;
    }
      
 
  };
  /**
     Values are added to vectors one by one sequentially.
     Those vectors are written as column vectors to the
     .m-file when this class is destroyed.

     @param VectorName name of vector to append value to
     @param value value to append at the end of VectorName

  */
  void addToVector(std::string VectorName, double value){
    name_to_doublevector[VectorName].push_back(value);
  };
  /**
     Adds additional strings to the end of the .m-file one by on. 
     Repeated calls to this functions adds new strings separated 
     by a new line below the previously added strings.
     Those strings could generally be commands, either to reshape 
     data into matrices, or perform automated test for the validity 
     of the data. All data vectors is written in the .m-file before 
     these additional commands.

     @param command string to append in the end of the m-file.
  */
  void appendString(std::string command){
    appended_commands.push_back(command);
  }
  
 private:
  std::string _filename;
  std::ofstream dest;
  std::map<std::string, std::vector<double> > name_to_doublevector;
  std::list<std::string> appended_commands;
};

#endif
