#ifndef MFILEREADER_HH
#define MFILEREADER_HH

#include <iostream>
#include <fstream>
#include <vector>
#include <map>
#include <cstdlib>
#include <sstream>

/**
   Used for reading numbers stored in vectors in .m-files.

@code
//Example usage of MFileReader, initialised to read from the file "results.m"

MFileReader m("results.m");

for(int i=0; i<m["x"].size(); i++){
  cout << "x["  << i << "]=" << m["x"][i] << ", "
       << "xx[" << i << "]=" << m["xx"][i] << endl;
}
vector<double> x = m["x"]; //to improve performance, get access 
                           //to the vector directly, instead of
                           //having to look up the vector
                           //through m["x"] during each access.
@endcode


   2006-02-22:
       When the .m-file being parsed contains exactly this line<br>
       "%====[end of data (MFileReader won't read past this line)]===="
       <br>
       thiss class stops parsing the .m-file for additional vectors.
   @author Simon Gustafsson (www.simong.se)
*/
class MFileReader{
 public:
  void print(){
    typedef std::map<std::string, std::vector<double> >::const_iterator CI;
    for(CI p=name_to_doublevector.begin(); p!=name_to_doublevector.end(); p++){
      std::cout << p->first << " = [";
      
      for(int i=0; i<((int)p->second.size()-1); i++){
	std::cout << p->second[i] << ";" << std::endl;
      }
      std::cout << p->second[p->second.size()-1] << "];" << std::endl << std::endl;
    }
  };
 private:
  void parser_remove_junk(){
    static const char* stopString = "%====[end of data (MFileReader won't read past this line)]====";
    char c;
    std::string unused;
    while(source.get(c)){
      if(isspace(c)) continue;
      if(c=='%') { 

	//Check for mark not to read past...
	int index=0;
	while(c==stopString[index]){
	  index++;
	  if(stopString[index]==0) {
	    reached_stopString=1;
	    break; //Entire string matched. Wont care about junk at end of line
	  }
	  source.get(c);
	}
	  
	while(source.get(c)){
	  if(c=='\n') break;
	}
	continue; 
      }
      source.putback(c);
      break;
    }
  };
  std::string parseIdentifier2(){
    char c;
    std::string name("");
    while(source.get(c)){
      if(isalnum(c) || (c=='_')){
	name = name + c;
	continue;
      }
      source.putback(c);
      return name;
    }
    return name;
  };
  std::string parseIdentifier(){
    char c;
    source.get(c);
    std::string name = c + parseIdentifier2();
    return name;
  };
  /**@todo should we allow empty matrices...*/
  void parseData(std::string identifier){
    double value;
    char c;
    while( source >> value){
      //std::cerr << value << std::endl;
      name_to_doublevector[identifier].push_back(value);
      parser_remove_junk();
      source.get(c);
      if( (c==';') || (c==',') ){ parser_remove_junk(); continue; }
      source.putback(c);
    }
    source.clear(); //Reset stream flags used in reading
    parser_remove_junk();
  };
  void parseVector(){
    char c;
    std::string identifier = parseIdentifier();
    parser_remove_junk();
    source.get(c);
    if(c!='=') throw "Error parsing expression. \"=\" expected";
    parser_remove_junk();
    source.get(c);
    if(c!='[') throw "Error parsing expression. \"[\" expected";
    parser_remove_junk();
    parseData(identifier);
    source.get(c);
    if(c!=']') throw "Error parsing expression. \"]\" expected";
    parser_remove_junk();
    source.get(c);
    if(c!=';') { source.putback(c); }
    std::cerr << "parsed vector \"" << identifier << "\"." << std::endl;
  };
  void parseMFile(){
    parser_remove_junk();
    if(source && !reached_stopString){
      //NOT EOF --> a Vector
      parseVector();
      parseMFile();
    }else{
      //EOF
      return;
    }
  };
 public:
  
  MFileReader(std::string filename): _filename(filename), source(_filename.c_str()),reached_stopString(0){
    
    if(!source) { 
      std::cerr << "Unable to open " << _filename << " for reading..." << std::endl; 
      /**@todo FIXME throw some kind of exception???*/
    }else {
      parseMFile();
    }
    
  };
  ~MFileReader(){
  };

  std::vector<double>& operator[](const std::string& index) {
    //2005-11-11: previously return name_to_doublevector[index]
 
    std::map<std::string, std::vector<double> >::iterator p;
    p = name_to_doublevector.find(index); 

    if(p==name_to_doublevector.end()){
        std::ostringstream s;
        s << "Trying to access vector \"" << index << "\" not stored in m-file \""
          << _filename << "\"";
          std::cerr << s << std::endl << std::flush; //FIXME won't flush before program terminated!
        throw s.str(); //FIXME should we throw this kind of exception
        //return ?;
    }else{
        return p->second;
    }
    
  }
  
/* not needed, so not finished. Should make toreturn a private var in class,
   and loaded while parsing data, or directly after. then return const reference to it
   
  std::vector<std::string>& getNames(){
    std::vector<std::string> toreturn;
    
    std::map<std::string, std::vector<double> >::const_iterator p;
    for( p=name_to_doublevector.begin(); p!= name_to_doublevector.end(); p++){
        toreturn.push_back(p->first);
    }
    return toreturn;
  }
*/  
 private:
  std::map<std::string, std::vector<double> > name_to_doublevector;
  std::string _filename;
  std::ifstream source;
  int reached_stopString;
};

#endif
